## ASTM Sender
## By Philip Hardy
## Public Domain

import PySimpleGUI as sg
import socket
import os
import time
import ipaddress
import json
import string
from operator import itemgetter

# Define globals
conn = False
addr = False
s = False
connection = False
sending = False
waitSend = False
nothingReceived = time.time()
defaultIP1 = ["", "", "Default IP 1"]
defaultIP2 = ["", "", "Default IP 2"]
files = []
nextFile = None
chain = False
waitChain = False
ASTMFilePath = ""
chainTime = time.time()
chainWaitDelay = 5
Stop = False
autoSend = True

# Load configurable data for Default IP buttons if available (JSON file)
try:
    with open ("conf.json", "r") as confFile:
        conf = json.load(confFile)
except Exception: # I don't care why it failed (probably FileNotFound) just set some blank defaults instead
    confFile = '{"defaultIP1": ["", "", "Default IP 1"], "defaultIP2": ["", "", "Default IP 2"]}'
    conf = json.loads(confFile)

# Put each IP and description in its on list
defaultIP1 = conf["defaultIP1"]
defaultIP2 = conf["defaultIP2"]

# Crack on...
def main():
    global connection, sending, waitSend, mainWindow, ASTMFilePath, nextFile, waitChain, chain, Stop
    localIP = getLocalIP()

    sg.theme("Default1") # A sensible look and feel. I supose this could be configurable.

    # Define pop up menu
    rightClickMenuDef = [[], ["About ASTM Sender", "Configure Default IP Buttons"]]

    # Define window layouts
    mainLayout = defMainLayout(localIP)

    confLayout = defConfLayout()

    infoLayout = defInfoLayout()

    # Open the main window
    mainWindow, confWindow, infoWindow = sg.Window('ASTM Sender', mainLayout, right_click_menu = rightClickMenuDef, resizable=True, finalize = True), None, None

    # The Main Event (loop)
    while True:
        window, event, values = sg.read_all_windows()

        # What to do if various windows are closed
        if event == sg.WINDOW_CLOSED:
            window.close() # Close the window first
            if window == confWindow:
                confWindow = None # Nothing
            elif window == infoWindow:
                infoWindow = None # Nothing
            elif window == mainWindow:
                break # Exit

        # Pop up menu click first
        if event == "About ASTM Sender" and not infoWindow:
            infoLayout = defInfoLayout()
            infoWindow = sg.Window("", infoLayout, finalize=True)

        if event == "Configure Default IP Buttons" and not confWindow:
            confLayout = defConfLayout()
            confWindow = sg.Window("Configure Default IP Buttons", confLayout, finalize=True)

        # Config window button clicks
        if event == "Save":
            saved = saveConf(values["DefaultIP1IP"], values["DefaultIP1Port"], values["DefaultIP1Desc"], values["DefaultIP2IP"], values["DefaultIP2Port"], values["DefaultIP2Desc"], values["AutoSend"], values["ChainWaitDelay"])
            if saved == True:
                window.close()
                confWindow = None

        if event == "Cancel":
            window.close()
            confWindow = None

        # Main window button clicks/events
        if event == "Load File" or event == "Load Folder":
            if event == "Load File": ASTMFilePath = sg.popup_get_file("Select Message File")
            if event == "Load Folder": ASTMFilePath = sg.popup_get_folder("Select folder that contains message files named 'ASTM_Message_File_<number>'.")
            if ASTMFilePath != None:
                ASTMFile = getASTMFile(ASTMFilePath)
                if chain == True: updateConsole(mainWindow, "**** CHAINED MESSAGES MODE ****", "#ff9000")
                mainWindow["ASTMBox"].update("")
                if ASTMFile != "!INVALID":
                    displayASTM = ""
                    displayASTM = stripCR(ASTMFile)
                    for line in displayASTM:
                        mainWindow["ASTMBox"].print(line)

        if event == "Client":
            if connection == False:
                mainWindow.perform_long_operation(lambda : openConnection(values["IP"], int(values["Port"]), mainWindow), "OpenComplete")

        if event == "Server":
            if connection == False:
                mainWindow.perform_long_operation(lambda : listen(values["IP"], int(values["Port"]), mainWindow), "ListenComplete")

        if event == "Send Message":
            Stop = False
            ASTMToSend, displayASTM = prepareMessageToSend(mainWindow, values)
            if ASTMToSend != "!INVALID":
                if sending == False:
                    sending = True
                    waitChain = False
                    mainWindow.perform_long_operation(lambda : sendMessage(mainWindow, values, ASTMToSend, displayASTM), "SMComplete")
                else:
                    waitSend = True
                    updateConsole(mainWindow, "## Waiting for socket to become available", "#ff0000")
                    ##updateConsole(mainWindow, "## waitSend="+str(waitSend)+" connection="+str(connection)+" sending="+str(sending), "#ff0000")

        if event == "Stop Sending":
            updateConsole(mainWindow, "## Stopped.", "#ff0000")
            if chain == True:
                updateConsole(mainWindow, "Pressing Send Message will resume sending chained messages", "#ff9000")
            Stop = True
            waitSend = False
            waitChain = False

        if event == "ASTMBox":
            mainWindow["ToSend"].update("Message to send: Edited")
            chain = False

        if event == "Close Connection":
            closeConnection(mainWindow)

        if event == "Save Console":
            saveConsole(mainWindow, values)

        if event == "DIP1":
            mainWindow["IP"].update(defaultIP1[0])
            mainWindow["Port"].update(defaultIP1[1])

        if event == "DIP2":
            mainWindow["IP"].update(defaultIP2[0])
            mainWindow["Port"].update(defaultIP2[1])

        if event == "LIP":
            localIP = getLocalIP()
            mainWindow["IP"].update(localIP)
            mainWindow["Port"].update("49999")

        # Complete threads events
        if event == "SMComplete":
            mainWindow["Send Message"].update(disabled=False)
            if chain == False: mainWindow["Stop Sending"].update(disabled=True)
            sending = False
            waitChain = False
            if chain == True and Stop == False:
                nextFile += 1
                ASTMFile = loadNextMessage(ASTMFilePath, chainWaitDelay)
                if ASTMFile != "":
                    waitChain = True
                    chainTime = time.time()
                    mainWindow["ASTMBox"].update("")
                    if ASTMFile != "!INVALID":
                        displayASTM = ""
                        displayASTM = stripCR(ASTMFile)
                        for line in displayASTM:
                            mainWindow["ASTMBox"].print(line)

        if event == "PCComplete":
            if waitChain == False: mainWindow["Stop Sending"].update(disabled=True)
            sending = False

        # Things to do that are not directly based on the event
        # If an message is waiting and the socket has become availale, send it.
        if waitSend == True and connection == True and sending == False:
            sending = True
            waitSend = False
            mainWindow.perform_long_operation(lambda : sendMessage(mainWindow, values, ASTMToSend, displayASTM), "SMComplete")

        cwd = int(chainWaitDelay)
        if chainWaitDelay == None: cwd = 0
        if waitChain == True and Stop == False and autoSend == True and time.time() > (chainTime + cwd):
            ASTMToSend, displayASTM = prepareMessageToSend(mainWindow, values)
            if ASTMToSend != "!INVALID":
                if sending == False:
                    sending = True
                    waitChain = False
                    mainWindow.perform_long_operation(lambda : sendMessage(mainWindow, values, ASTMToSend, displayASTM), "SMComplete")
                else:
                    waitSend = True
                    updateConsole(mainWindow, "## Waiting for socket to become available", "#ff0000")

        # If the connection is open and we're not otherwise busy, check for incoming data.
        if connection == True and sending == False:
            sending = True
            mainWindow.perform_long_operation(lambda : pollConnection(mainWindow, values), "PCComplete")

    mainWindow.close()

def defMainLayout(localIP):

    mainLayout = [
                    [sg.Button("Load Message", key="Load File"), sg.Button("Load Folder", key="Load Folder"), sg.Push(), sg.Text("Save console to:"), sg.Input(), sg.FileSaveAs("Choose Location", key="SaveAs"), sg.Button("Save Console")],
                    [sg.Push(), sg.Radio("Normal ASTM", group_id="Message Type", default=True, key = "ASTM"), sg.Radio("Compressed ASTM", group_id="Message Type", default=False, key = "Compressed"), sg.Radio("XML over ASTM", group_id="Message Type", default=False, key = "XML"), sg.Radio("BN-II Protocol", group_id="Message Type", default=False, key = "BN-II"), sg.Radio("HM-JACKarc Protocol", group_id="Message Type", default=False, key = "HMJACKarc")],
                    [sg.Text("Message to send:", key="ToSend")],
                    [sg.Multiline("", size=(130,15), font="Consolas 10", expand_x=True, expand_y=True, horizontal_scroll=True, enable_events=True, key="ASTMBox")],
                    [sg.Multiline("## ASTM Sender V0.01 Sun 18 Dec 2022 by Phil Hardy\n", size=(130,20), font="Consolas 10", horizontal_scroll=False, expand_x=True, expand_y=True, disabled=True, autoscroll=True, background_color="#000000", text_color="#ff0000", key="Console")],
                    [sg.Text("IP address:"), sg.Input(key="IP", default_text=localIP, size=(15, 1)), sg.Text("Port:"), sg.Input(key="Port", default_text="49999", size=(5, 1)), sg.Push(), sg.Button(defaultIP1[2], key="DIP1", tooltip="Right click to customise this button"), sg.Button(defaultIP2[2], key="DIP2", tooltip="Right click to customise this button"), sg.Button("Use Local IP", key="LIP")],
                    [sg.Button("Listen For Analyser (Act as Server)", key="Server"), sg.Button("Connect To Analyser (Act as Client)", key="Client"), sg.Push(), sg.Button("Stop Sending", disabled=True), sg.Button("Send Message", disabled=True), sg.Button("Close Connection", disabled=True)]  ]
    return(mainLayout)

def defConfLayout():
    confLayout = [
                    [sg.Text("Default IP 1:"), sg.Input(key="DefaultIP1IP", default_text=defaultIP1[0], size=(15,1)), sg.Text("Port:"), sg.Input(key="DefaultIP1Port", default_text=defaultIP1[1], size=(5,1)), sg.Text("Description:"), sg.Input(key="DefaultIP1Desc", default_text=defaultIP1[2], size=(20,1))],
                    [sg.Text("Default IP 2:"), sg.Input(key="DefaultIP2IP", default_text=defaultIP2[0], size=(15,1)), sg.Text("Port:"), sg.Input(key="DefaultIP2Port", default_text=defaultIP2[1], size=(5,1)), sg.Text("Description:"), sg.Input(key="DefaultIP2Desc", default_text=defaultIP2[2], size=(20,1))],
                    [sg.Checkbox("Automatically send next chained message after delay", key="AutoSend", default=autoSend), sg.Push(), sg.Text("Delay (secs):"), sg.Input(key="ChainWaitDelay", default_text=chainWaitDelay, size=(3,1))],
                    [sg.Button("Save"), sg.Button("Cancel")]  ]
    return(confLayout)

def defInfoLayout():
    infoLayout = [
                    [sg.Image("research.png", size=(256,64), expand_x=True)],
                    [sg.Text("ASTM Sender", justification="center", expand_x=True, font="Default 13 bold")],
                    [sg.Text("Version 0.01 (Sun 18 Dec 2022) Public Domain", justification="center", expand_x=True)],
                    [sg.Text("By Philip Hardy", justification="center", expand_x=True, font="Default 13")],
                    [sg.Text("Lab icons created by Freepik - Flaticon https://www.flaticon.com/free-icons/lab", justification="center", expand_x=True, font="Default 6")]  ]
    return(infoLayout)

def saveConf(ip1, ip1Port, ip1Desc, ip2, ip2Port, ip2Desc, chainWaitCheckBox, chainWaitBox):
    global defaultIP1, defaultIP2, chainWaitDelay, autoSend
    if chainWaitCheckBox == False:
        chainWaitDelay = None
        autoSend = False
    else:
        if chainWaitBox == "": chainWaitBox = 5
        chainWaitDelay = chainWaitBox
        autoSend = True
    if validateIPAddress(ip1) == True and validateIPAddress(ip2) == True:
        conf = {"defaultIP1": [ip1, ip1Port, ip1Desc], "defaultIP2": [ip2, ip2Port, ip2Desc]}
        with open("conf.json", "w") as confFile:
            json.dump(conf, confFile)
        mainWindow["DIP1"].update(ip1Desc)
        mainWindow["DIP2"].update(ip2Desc)
        defaultIP1 = [ip1, ip1Port, ip1Desc]
        defaultIP2 = [ip2, ip2Port, ip2Desc]
    else:
        if validateIPAddress(ip1) == False and validateIPAddress(ip2) == False:
            sg.popup("Both default IP addresses not valid")
            return(False)
        if validateIPAddress(ip1) == False:
            sg.popup("Default IP 1 not valid")
            return(False)
        if validateIPAddress(ip2) == False:
            sg.popup("Default IP 2 not valid")
            return(False)
    return(True)

def getLocalIP():
    try:
        localIP = socket.gethostbyname_ex(socket.gethostname())[-1][-1]
    except:
        localIP = "127.0.0.1"
    return(localIP)

def saveConsole(mainWindow, values):
    fileExists = False
    if values["SaveAs"] == "":
        sg.popup("No location to save console file to selected.")
        return()
    outputLocation = values["SaveAs"]
    with open(outputLocation, "wt", encoding = "ascii") as consoleFile:
        consoleFile.write(mainWindow["Console"].get())
    updateConsole(mainWindow, "## Console text saved to: " + values["SaveAs"], "#ff0000")
    return()

def closeConnection(mainWindow):
    global connection, conn, addr, s
    if conn:
        conn.close()
    if s:
        s.close()
    connection = False
    updateConsole(mainWindow, "## Connection closed.", "#ff0000")
    mainWindow["Server"].update(disabled=False)
    mainWindow["Client"].update(disabled=False)
    mainWindow["Send Message"].update(disabled=True)
    mainWindow["Stop Sending"].update(disabled=True)
    mainWindow["Close Connection"].update(disabled=True)
    return()

def pollConnection(mainWindow, values):
    global connection, conn, addr, polltime, nothingReceived
    polltime = time.time()
    if polltime > (nothingReceived + 60):
        updateConsole(mainWindow, "## Nothing received for 60 seconds.", "#ff0000")
        nothingReceived = polltime
    mainWindow.refresh()
    incoming = ""
    received = ""
    completeMessage = False
    messageTimeout = time.time() + 2
    while completeMessage == False:
        try:
            conn.settimeout(1)
            time.sleep(0.225)
            incoming = conn.recv(2048)
        except Exception as ex:
            if str(ex) == "timed out":
                if received == "":
                    return()
            else:
                conn.close()
                connection = False
                return()
        if not incoming:
            conn.close()
            connection = False
            updateConsole(mainWindow, "## Connection closed.", "#ff0000")
            mainWindow["Server"].update(disabled=False)
            mainWindow["Client"].update(disabled=False)
            mainWindow["Send Message"].update(disabled=True)
            mainWindow["Close Connection"].update(disabled=True)
            return()
        nothingReceived = time.time()
        try:
            received = received + incoming.decode("ascii")
            displayReceived = showControlChar(received)
        except Exception as ex:
            displayReceived = str(ex)
        lastCharacter = ord(received[len(received)-1])
        currentTime = time.time()
        if  lastCharacter == 10 or lastCharacter == 3 or lastCharacter == 23 or lastCharacter == 5 or lastCharacter == 4 or lastCharacter == 6:
            completeMessage == True
            break
        if currentTime > messageTimeout:
            break
    displayReceived = ''.join(displayReceived)
    updateConsole(mainWindow, "<- " + displayReceived, "#009000")
    lr = len(received)
    if lr > 1 and ord(received[0]) == 4 and ord(received[lr-1]) == 5:
        conn.send(chr(6).encode("ascii")) # Then send <ACK>
        updateConsole(mainWindow, "-> <ACK>", "#00ff00")
    if lr == 1 and ord(received) == 5: # If length of message is 1 char and is <ENQ>
        conn.send(chr(6).encode("ascii")) # Then send <ACK>
        updateConsole(mainWindow, "-> <ACK>", "#00ff00")
    if ord(received[lr-1]) == 10: # If message ends with <LF>
        if ord(received[0]) == 2: # And if message starts with <STX>
            if ((ord(received[lr-5]) == 3 or ord(received[lr-5]) == 23)) and (values["ASTM"] == True or values["Compressed"] == True or values["XML"] == True): # And we're expecting ASTM and the message finished with <ETB> or <ETX>
                conn.send(chr(6).encode("ascii")) # Then send <ACK>
                updateConsole(mainWindow, "-> <ACK>", "#00ff00")
            elif values["BN-II"] == True: # Or if we're expecting a BN-II message
                conn.send(chr(6).encode("ascii")) # Then send <ACK>
                updateConsole(mainWindow, "-> <ACK>", "#00ff00")
            else:
                conn.send(chr(21).encode("ascii")) # Otherwise Send <NAK>
                updateConsole(mainWindow, "-> <NAK>", "#00ff00")
    elif ord(received[lr-1]) == 3 and values["HMJACKarc"] == True: # Or if it ends with <ETX> and we're expecting an HM-JACKarc message
        conn.send((chr(2)+chr(6)+chr(3)).encode("ascii")) # Then send <STX><ACK><ETX>
        updateConsole(mainWindow, "-> <STX><ACK><ETX>", "#00ff00")
    return()

def validateIPAddress(address):
    try:
        ipOrHost = socket.gethostbyname(address)
        return(True)
    except socket.gaierror as ex:
        try:
            ip = ipaddress.ip_address(address)
            return(True)
        except ValueError:
            return(False)
        except Exception as ex:
            debugInfo(ex)
            raise
    except Exception as ex:
        debugInfo(ex)
        raise
    return(False)

def stripCR(string):
    strippedString = []
    for sub in string:
        strippedString.append(sub.replace("\n", ""))
    return(strippedString)

def checkSum(preparedLine):
    checkSum = (hex(sum(preparedLine.encode("ascii")) % 256))
    if len(checkSum) == 3:
        checkSum = checkSum[0] + "0" + checkSum[2]
    checkSum = checkSum[-2:]
    checkSum = checkSum.upper()
    return(checkSum)

def checkSumBNII(preparedLine):
    checkSum = 0
    for char in preparedLine:
        checkSum += (ord(char))
    checkSum = 64 - (checkSum % 64)
    if checkSum < 32:
        checkSum += 64
    checkSum = chr(checkSum)
    return(checkSum)

def prepareMessageToSend(mainWindow, values):
    charsRead = ""
    for line in values["ASTMBox"]:
        charsRead = charsRead + line
    linesRead = charsRead.split("\n")
    if values["ASTM"] == True or values["Compressed"]:
        valid = validateASTM(linesRead)
    if values["XML"] == True:
        valid = validateXMLOverASTM(linesRead)
    if values["BN-II"] == True:
        valid = validateBNII(linesRead)
    if values["HMJACKarc"] == True:
        valid = validateHMJACKarc(linesRead)
    if valid == "!INVALID":
        linesRead = "!INVALID"
        return("!INVALID", "")
    if values["Compressed"] == True:
        linesRead = stripCR(linesRead)
        linesRead = prepareCompressed(linesRead)
    elif values["ASTM"] == True:
        linesRead = stripCR(linesRead)
        linesRead = prepareUncompressed(linesRead)
    elif values["BN-II"] == True:
        linesRead = prepareBNII(linesRead)
    elif values["HMJACKarc"] == True:
        linesRead = stripCR(linesRead)
        linesRead = prepareHMJACKarc(linesRead)
    displayMessage = showControlChar(linesRead)
    if valid != "!INVALID":
        updateConsole(mainWindow, "## Prepared message to send:", "#ff0000")
        for line in displayMessage:
            updateConsole(mainWindow, "'" + line + "'", "#ffff00")
    return(linesRead, displayMessage)

def validateXMLOverASTM(rawFile):
    return(rawFile)

def validateBNII(rawFile):
    return(rawFile)

def validateHMJACKarc(rawFile):
    invalid = False
    messageType = None
    for line in rawFile:
        if line[-1] != ",":
            invalid = True
            sg.popup("Invalid HMJACKarc Message Detected.", line)
            break
    if invalid == False:
        if rawFile[0].startswith("2,"):
            messageType = "Normal"
            updateConsole(mainWindow, "## HMJACKarc message type: Normal", "#ff0000")
        elif rawFile[0].startswith("21,"):
            messageType = "Compatible 1"
            updateConsole(mainWindow, "## HMJACKarc message type: Compatible 1", "#ff0000")
        elif rawFile[0].startswith(tuple(string.digits)):
            if len(rawFile[0]) > 2:
                if rawFile[0][3] == "," and len(rawFile[0]) > 41:
                    messageType = "Compatible 2 or 3"
                    updateConsole(mainWindow, "## HMJACKarc message type: Compatible 2 or 3", "#ff0000")
                elif rawFile[0][3] == "," and len(rawFile[0]) < 41:
                    messageType = "Order Confirmation"
                    updateConsole(mainWindow, "## HMJACKarc message type: Order Confirmation", "#ff0000")
                else:
                    invalid = True
                    sg.popup("Invalid HMJACKarc Message. Message Type could not be detected. ")
        else:
            invalid = True
            sg.popup("Invalid HMJACKarc Message. Message Type could not be detected. ")
    if invalid == True:
        rawFile = "!INVALID"
    return(rawFile)

def validateASTM(rawFile):
    for line in rawFile:
        valid = line.startswith(("H|", "P|", "O|", "L|", "R|", "C|", "Q|", "M|", "S|"))
        if valid == False:
            valid = line.startswith(("H|", "P|", "O|", "L|", "R|", "C|", "Q|", "M|", "S|"), 1)
            if valid == False:
                sg.popup("Invalid ASTM Detected:", line)
                rawFile = "!INVALID"
                break
    return(rawFile)

def prepareBNII(bnIILines):
    preparedLines = []
    for line in bnIILines:
        checkDigit = checkSumBNII(line)
        preparedLine = chr(2) + line + checkDigit + chr(13) + chr(10)
        preparedLines.append(preparedLine)
    return(preparedLines)

def prepareHMJACKarc(HMJACKarcLines):
    preparedLines = []
    for line in HMJACKarcLines:
        preparedLine = chr(2) + line + chr(3)
        preparedLines.append(preparedLine)
    return(preparedLines)

def prepareUncompressed(astmLines):
    inLine = 0
    outLine = 0
    ASTMNo = 1
    endChar = 3 ## <ETX>
    preparedLines = []
    while inLine < len(astmLines):
        if len(astmLines[inLine]) > 238:
            longLine = []
            while len(astmLines[inLine]) > 238:
                longLine.append(astmLines[inLine][0:238])
                astmLines[inLine] = astmLines[inLine][239:len(astmLines[inLine])]
            longLine.append(astmLines[inLine])
            numberOfLongLines = len(longLine)
            line = 0
            endChar = 23 ## <ETB>
            while (line + 1) <= numberOfLongLines:
                if ASTMNo == 8:
                    ASTMNo = 0
                if (line + 1) == numberOfLongLines:
                    endChar = 3 ## <ETX>
                preparedLine = str(ASTMNo) + longLine[line] + chr(endChar)
                preparedLine = chr(2) + preparedLine + checkSum(preparedLine) + chr(13) + chr(10)
                preparedLines.append(preparedLine)
                ASTMNo +=1
                outLine +=1
                line +=1
        else:
            if ASTMNo == 8:
                ASTMNo = 0
            preparedLine = str(ASTMNo) + astmLines[inLine] + chr(13) + chr(3)
            preparedLine = chr(2) + preparedLine + checkSum(preparedLine) + chr(13) + chr(10)
            preparedLines.append(preparedLine)
            outLine +=1
            ASTMNo +=1
        inLine +=1
    return(preparedLines)

def prepareCompressed(linesRead):
    line = 0
    ASTMNo = 1
    compressedLine = ""
    while line < len(linesRead):
        compressedLine = compressedLine + linesRead[line] + chr(13)
        line +=1
    linesRead = []
    while len(compressedLine) > 238:
        linesRead.append(compressedLine[0:238])
        compressedLine = compressedLine[239:len(compressedLine)]
    linesRead.append(compressedLine)
    numberOfLines = len(linesRead)
    if numberOfLines == 1:
        linesRead[0] = str(ASTMNo) + linesRead[0] + chr(3)
        linesRead[0] = chr(2) + linesRead[0] + checkSum(linesRead[0]) + chr(13) + chr(10)
    elif numberOfLines > 1:
        line = 0
        endc = 23
        while (line + 1) <= numberOfLines:
            if ASTMNo == 8:
                ASTMNo = 0
            if (line + 1) == numberOfLines:
                endc = 3
            linesRead[line] = str(ASTMNo) + linesRead[line] + chr(endc)
            linesRead[line] = chr(2) + linesRead[line] + checkSum(linesRead[line]) + chr(13) + chr(10)
            ASTMNo +=1
            line +=1
    return(linesRead)

def updateConsole(mainWindow, line, colour):
    time.ctime()
    mainWindow["Console"].print(time.strftime("%H:%M:%S "), text_color = "#ffffff", end="")
    mainWindow["Console"].print(line, text_color = colour)
    mainWindow.refresh()

def showControlChar(string):
    stringBack = []
    controlChars = ("<NUL>","<SOH>","<STX>","<ETX>","<EOT>","<ENQ>","<ACK>","<BEL>","<BS>","<TAB>","<LF>","<VT>","<FF>","<CR>","<SO>","<SI>","<DLE>","<DC1>","<DC2>","<DC3>","<DC4>","<NAK>","<SYN>","<ETB>","<CAN>","<EM>","<SUB>","<ESC>","<FS>","<GS>","<RS>","<US>")
    listLength = len(string)
    for line in range(listLength):
        checkLine = string[line]
        stringLength = len(checkLine)
        lineBack = ""
        for char in range(stringLength):
            checkChar = checkLine[char]
            if ord(checkChar) < 32:
                lineBack = lineBack + controlChars[ord(checkChar)]
            elif ord(checkChar) == 127:
                lineBack = lineBack + "<DEL>"
            else:
                lineBack = lineBack + checkChar
        stringBack.append(lineBack)
    return(stringBack)

def getASTMFile(ASTMPath):
    global files, nextFile, chain
    rawASTM = ""
    fileExists = False
    if ASTMPath == "" or str(ASTMPath) == "None":
        ASTMPath = ""
    else:
        fileExists = os.path.exists(ASTMPath) # Check file path is valid
    if fileExists == False:
        sg.popup("File not found")
        rawFile = ""
    else:
        directory = os.path.isdir(ASTMPath)
        if directory == True:
            files = os.listdir(ASTMPath)
            files.sort(key=itemgetter(slice(19, None)))
            nextFile = 0
            chain = True
            noOfFiles = len(files)
            while files[nextFile].startswith("ASTM_Test_Message") != True:
                nextFile += 1
                if nextFile >= noOfFiles:
                    updateConsole(mainWindow, "Folder does not contain any files named 'ASTM_Test_Message_<number>'.", "#ff0000")
                    rawFile = ""
                    chain = False
                    return(rawFile)
            if ASTMPath.endswith("/"):
                loadASTMFile = open(ASTMPath+files[nextFile], "r")
            else:
                loadASTMFile = open(ASTMPath+"/"+files[nextFile], "r")
            rawFile = loadASTMFile.readlines()
            mainWindow["ToSend"].update("Message To Send: " + files[nextFile] + " (CHAINED)")
            loadASTMFile.close()
        else:
            chain = False
            loadASTMFile = open(ASTMPath, "r") # Open file
            rawFile = loadASTMFile.readlines() # Read each line into list
            loadASTMFile.close()
            mainWindow["ToSend"].update("Message To Send: " + ASTMPath)
            ##rawASTM = validateASTM(rawFile)
            ##if rawASTM == "":
            ##sg.popup("Unable to load ASTM file")
    return(rawFile)

def stopChain():
    global chain, Stop, waitChain, waitSend
    chain = False
    Stop = True
    waitChain = False
    waitSend = False
    return()

def loadNextMessage(ASTMPath, chainWaitDelay):
    global chain, Stop, waitChain, waitSend
    if nextFile >= len(files):
        stopChain()
        updateConsole(mainWindow, "Finished sending all messages.", "#ff9000")
        return("")
    try:
        if os.path.isdir(ASTMPath) == False:
            stopChain()
            updateConsole(mainWindow, "File path changed: No longer a folder. Stopping.", "#ff9000")
            return("")
    except TypeError as ex:
        debugInfo(ex)
        updateConsole(mainWindow, "File path not a string.", "#ff9000")
        stopChain()
        return("")
    except Exception as ex:
        debugInfo(ex)
        stopChain()
        return("")
    if ASTMPath.endswith("/"):
        loadASTMFile = open(ASTMPath+files[nextFile], "r")
    else:
        loadASTMFile = open(ASTMPath+"/"+files[nextFile], "r")
    rawFile = loadASTMFile.readlines()
    loadASTMFile.close()
    mainWindow["ToSend"].update("Message To Send: " + files[nextFile] + "(CHAINED)")
    updateConsole(mainWindow, "Loaded next chained file: " + str(files[nextFile]), "#ff9000")
    if chainWaitDelay != None:
        updateConsole(mainWindow, "Waiting " + str(chainWaitDelay) + " seconds before sending...", "#ff9000")
    else:
        updateConsole(mainWindow, "Click 'Send Message' when ready to continue.", "#ff9000")
    return(rawFile)

def listen(HOST, PORT, mainWindow):
    global connection, conn, addr, nothingReceived, s
    connection = False
    HOST = getLocalIP()
    validIP = validateIPAddress(HOST)
    if validIP == False:
        updateConsole(mainWindow, "## IP Address not valid.", "#ff0000")
        sg.popup("IP address invalid")
        return()
    try:
        if socket.has_dualstack_ipv6() == True:
            s = socket.create_server((HOST,PORT), family=socket.AF_INET6, dualstack_ipv6=True)
        else:
            s = socket.create_server((HOST,PORT))
        s.bind((HOST, PORT))
    except OSError:
        try:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            s.bind((HOST, PORT))
        except Exception as ex:
            updateConsole(mainWindow, "## Cannot open port " + str(PORT) + ".", "#ff0000")
            debugInfo(ex)
            return()
    except Exception as ex:
        updateConsole(mainWindow, "## Cannot open port " + str(PORT) + ".", "#ff0000")
        debugInfo(ex)
        return()
    connection = True
    s.settimeout(30)
    updateConsole(mainWindow, "## Listening on port " + str(PORT) + ".", "#ff0000")
    mainWindow["Close Connection"].update(disabled=False)
    s.listen()
    try:
        conn, addr = s.accept()
    except TimeoutError as ex:
        if connection == True:
            updateConsole(mainWindow, "## No connection attempt on port " + str(PORT) + ".", "#ff0000")
            debugInfo(ex)
        mainWindow["Close Connection"].update(disabled=True)
        connection = False
        s.close()
        return()
    except Exception as ex:
        debugInfo(ex)
        mainWindow["Close Connection"].update(disabled=True)
        connection = False
        s.close()
        return()
    updateConsole(mainWindow, "## Connection accepted from " + str(addr) + " on port " + str(PORT) + ".", "#ff0000")
    mainWindow["Server"].update(disabled=True)
    mainWindow["Client"].update(disabled=True)
    mainWindow["Send Message"].update(disabled=False)
    mainWindow["Close Connection"].update(disabled=False)
    nothingReceived = time.time()
    return()

def openConnection(HOST, PORT, mainWindow):
    global connection, conn, addr, nothingReceived
    connection = False
    validIP = validateIPAddress(HOST)
    if validIP == False:
        updateConsole(mainWindow, "## IP Address not valid.", "#ff0000")
        return()
    ##conn = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    try:
        updateConsole(mainWindow, "## Opening connection to " + HOST + ":" + str(PORT) + ".", "#ff0000")
        conn = socket.create_connection((HOST, PORT), timeout=10, source_address=None, all_errors=False)
        ##conn.settimeout(10)
        ##conn.connect((HOST, PORT))
    except Exception as ex:
        updateConsole(mainWindow, "## Cannot open connection to " + HOST + ":" + str(PORT) + ".", "#ff0000")
        debugInfo(ex)
        return()
    updateConsole(mainWindow, "## Connected to " + HOST + ":" + str(PORT) + ".", "#ff0000")
    time.sleep(1)
    mainWindow["Server"].update(disabled=True)
    mainWindow["Client"].update(disabled=True)
    mainWindow["Send Message"].update(disabled=False)
    mainWindow["Close Connection"].update(disabled=False)
    connection = True
    nothingReceived = time.time()
    return()

def debugInfo(ex):
    updateConsole(mainWindow, "## "+ str(ex), "#ff0000")
    updateConsole(mainWindow, "!! DEBUG INFO: " + str(type(ex)) + " " + str(ex.args), "#808080")
    ## ConnectionRefusedError
    ## TimeoutError
    ## socket.gaierror
    return()

def sendMessage(mainWindow, values, ASTMToSend, displayASTM):
    global conn, addr, nothingReceived
    mainWindow["Send Message"].update(disabled=True)
    mainWindow["Stop Sending"].update(disabled=False)
    success = sendENQ(mainWindow)
    if success == True:
        count = 0
        for line in ASTMToSend:
            count += 1
            conn.send(line.encode("ascii"))
            if count <= len(displayASTM):
                updateConsole(mainWindow, "-> " + displayASTM[(count - 1)], "#00ff00")
            ACKBack = awaitACK(mainWindow, values)
            if Stop == True: break
            if ACKBack == False:
                updateConsole(mainWindow, "## <ACK> not received", "#ff0000")
                break
        sendEOT()
        updateConsole(mainWindow, "-> <EOT>", "#00ff00")
        nothingReceived = time.time()
    return()

def sendEOT():
    conn.send(chr(4).encode("ascii"))

def awaitACK(mainWindow, values):
    success = False
    try:
        incoming = conn.recv(1024)
    except:
        displayReceived = "## Timed out."
        incoming = False
    if not incoming:
        displayReceived = "## Connection closed."
    else:
        try:
            received = incoming.decode("ascii")
            displayReceived = showControlChar(received)
        except Exception as e:
            displayRecieved = str(e)
        updateConsole(mainWindow, "<- " + "".join(displayReceived), "#009000")
        if len(received) == 1 and ord(received[0]) == 6 and values["HMJACKarc"] == False:
            success = True
        elif ord(received[0]) == 2 and ord(received[1]) == 6 and (ord(received[2]) == 3) and values["HMJACKarc"] == True:
            success = True
    return(success)

def sendENQ(mainWindow):
    success = False
    timeout = time.time() + 6
    conn.send(chr(5).encode("ascii"))
    updateConsole(mainWindow, "-> <ENQ>", "#00ff00")
    while time.time() < timeout:
        incoming = conn.recv(1024)
        if not incoming:
            break
        try:
            received = incoming.decode("ascii")
            displayReceived = showControlChar(received)
        except Exception as e:
            displayReceived = str(e)
        updateConsole(mainWindow, "<- " + displayReceived[0], "#00e000")
        if ord(received) == 6:
            success = True
            break
        else:
            updateConsole(mainWindow, "## <ACK> not received", "#ff0000")
    return(success)

main() # Go back to the top because Python does things backwards

